unit fROR_Options;
{
================================================================================
*
*       Package:        ROR - Clinical Case Registries
*       Date Created:   $Revision: 17 $  $Modtime: 5/01/07 11:06a $
*       Site:           Hines OIFO
*       Developers:                        @domain.ext
*	Description:    This dialog controls application options
*
*	Notes:
*
================================================================================
*	$Archive: /CCR v1.5/Current/fROR_Options.pas $
*
* $History: fROR_Options.pas $
 * 
 * *****************  Version 17  *****************
 * User: Vhaishgavris Date: 5/01/07    Time: 3:20p
 * Updated in $/CCR v1.5/Current
 * 
 * *****************  Version 16  *****************
 * User: Vhaishgavris Date: 4/30/07    Time: 3:58p
 * Updated in $/CCR v1.5/Current
 * 
 * *****************  Version 15  *****************
 * User: Vhaishgavris Date: 3/08/07    Time: 1:47p
 * Updated in $/CCR v1.5/Current
 * 
 * *****************  Version 14  *****************
 * User: Vhaishgavris Date: 7/31/06    Time: 12:54p
 * Updated in $/CCR v1.5/Current
 * 
 * *****************  Version 13  *****************
 * User: Vhaishgavris Date: 11/16/05   Time: 10:30a
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 12  *****************
 * User: Vhaishgavris Date: 3/02/05    Time: 11:54a
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 11  *****************
 * User: Vhaishgavris Date: 3/01/05    Time: 4:54p
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 10  *****************
 * User: Vhaishgavris Date: 2/28/05    Time: 3:54p
 * Updated in $/CCR v1.0/Current
 *
 * *****************  Version 9  *****************
 * User: Vhaishgavris Date: 2/25/05    Time: 4:32p
 * Updated in $/CCR v1.0/Current
 *
 * *****************  Version 8  *****************
 * User: Vhaishgavris Date: 2/25/05    Time: 9:54a
 * Updated in $/CCR v1.0/Current
 *
================================================================================
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, ExtCtrls, ToolWin, ActnList, StdCtrls, Buttons, ImgList,
  uROR_CustomControls, uROR_ParamsPanel, ovcbase, ovcef, ovcpb, ovcnf,
  ovcsf, ovcfiler, ovcstore, uROR_VistAStore, ovcstate,
  VA508AccessibilityManager;

const
  ieReadOnly = 0;

type

  //--------------------------- TCCRInterfaceElement(s) ------------------------

  TCCRElementProps = set of ( cepColor, cepFontColor, cepFontSize );

  TCCRInterfaceElementDef = record
    //--- Auxiliary data
    Name:      String;
    Props:     TCCRElementProps;
    Descr:     String;
    //--- Stored properties
    Color:     TColor;
    FontColor: TColor;
    FontSize:  Integer;
  end;

  TCCRInterfaceElement = class(TCollectionItem)
  private
    fData: TCCRInterfaceElementDef;

    function  getColor: TColor;
    function  getDescription: String;
    function  getFontColor: TColor;
    function  getFontSize: Integer;
    function  getName: String;
    function  getProps: TCCRElementProps;
    procedure setColor(const aColor: TColor);
    procedure setFontColor(const aColor: TColor);
    procedure setFontSize(const aSize: Integer);

  public
    procedure AssignDef(const aData: TCCRInterfaceElementDef);
    procedure Restore(aStore: TOvcAbstractStore; const aSection: String);
    procedure Save(aStore: TOvcAbstractStore; const aSection: String);

    property Description: String          read    getDescription;
    property Name: String                 read    getName;
    property Props: TCCRElementProps      read    getProps;

  published
    property Color: TColor                read    getColor
                                          write   setColor;

    property FontColor: TColor            read    getFontColor
                                          write   setFontColor;

    property FontSize: Integer            read    getFontSize
                                          write   setFontSize;

  end;

  TCCRInterfaceElements = class(TCollection)
  protected
    function getItem(anIndex: Integer): TCCRInterfaceElement;

  public
    function  Add(const aData: TCCRInterfaceElementDef): TCCRInterfaceElement;
    procedure RestoreDefaults;

    property Items[Index: Integer]: TCCRInterfaceElement read getItem; default;

  end;

  //--------------------------------- TCCROptions ------------------------------

  TCCROptions = class(TPersistent)
  private
    fCDC1Bmp:           String;
    fCDC2Bmp:           String;
    fInterfaceElements: TCCRInterfaceElements;
    fMaxPatientCount:   Integer;
    fRPCTimeout:        Integer;
    fShowHints:         Boolean;

  public
    constructor Create;
    destructor  Destroy; override;

    procedure Apply; virtual;
    function  Edit: Boolean; virtual;
    procedure Restore(aState: TOvcPersistentState = nil); virtual;
    procedure RestoreDefaults; virtual;
    procedure Save(aState: TOvcPersistentState = nil); virtual;

  published
    property CDC1Bmp: String              read    fCDC1Bmp
                                          write   fCDC1Bmp;

    property CDC2Bmp: String              read    fCDC2Bmp
                                          write   fCDC2Bmp;

    property IE: TCCRInterfaceElements    read    fInterfaceElements;
                                          
    property MaxPatientCount: Integer     read    fMaxPatientCount
                                          write   fMaxPatientCount;

    property RPCTimeout: Integer          read    fRPCTimeout
                                          write   fRPCTimeout;

    property ShowHints: Boolean           read    fShowHints
                                          write   fShowHints;

  end;

  //-------------------------------- TFormOptions ------------------------------

  TFormOptions = class(TForm)
    ActionList1: TActionList;
    acDefault: TAction;
    pnlButtons: TPanel;
    btnDefaults: TButton;
    btnSave: TBitBtn;
    btnCancel: TBitBtn;
    pnlMain: TPanel;
    pcMain: TPageControl;
    tsGeneral: TTabSheet;
    pnlServer: TCCRParamsPanel;
    lblTimeout: TLabel;
    cmbxTimeout: TComboBox;
    edMaxPatientCount: TOvcSimpleField;
    vxsStorage: TCCRVistAStore;
    vxsOptions: TOvcPersistentState;
    tsAppearance: TTabSheet;
    pnlAppearance: TCCRParamsPanel;
    lblElement: TLabel;
    lblFontColor: TLabel;
    lblColor: TLabel;
    lblFontSize: TLabel;
    memoSample: TMemo;
    lbIE: TListBox;
    cmbxFontColor: TColorBox;
    cmbxColor: TColorBox;
    cmbxFontSize: TComboBox;
    CCRParamsPanel1: TCCRParamsPanel;
    cbShowHints: TCheckBox;
    VA508AccessibilityManager1: TVA508AccessibilityManager;
    stxtMaxPatientCount: TStaticText;
    procedure acDefaultExecute(Sender: TObject);
    procedure lbIEClick(Sender: TObject);
    procedure cmbxFontColorSelect(Sender: TObject);
    procedure cmbxColorSelect(Sender: TObject);
    procedure cmbxFontSizeChange(Sender: TObject);
    procedure vxsStorageBeforeOpen(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormShow(Sender: TObject);

  protected
    function  FetchControlValues(const Silent: Boolean = False): Boolean; virtual;
    procedure UpdateAllControls; virtual;
    procedure UpdateIEControls(const anIndex: Integer = -1);

  public
    constructor Create(AOwner: TComponent); override;
    procedure AfterConstruction; override;
  end;

const
  cepAll = [ cepColor, cepFontColor, cepFontSize ];

function CCROptions: TCCROptions;

implementation
{$R *.DFM}

uses
  uROR_Common, uROR_Broker, uROR_CommonUtils, uROR_Utilities, uROR_Strings,
  uROR_Classes;

const
  MINRPCTIMEOUT = 30;
  SECTION       = 'Options';

type
  TCCRApplyOptionsEvent = procedure of object;


var
  fCCROptions: TCCROptions = nil;

  InterfaceElementDefs: array [0..0] of TCCRInterfaceElementDef = (
    ( Name: 'ReadOnly';  Props: [cepColor, cepFontColor];
      Descr: 'Read-only Controls';
      Color: clInfoBk;  FontColor: clWindowText;  FontSize: 8 )
  );

function CCROptions: TCCROptions;
begin
  if fCCROptions = nil then
    fCCROptions := TCCROptions.Create;
  Result := fCCROptions;
end;

/////////////////////////////// TCCRInterfaceElement \\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRInterfaceElement.AssignDef(const aData: TCCRInterfaceElementDef);
begin
  fData := aData;
end;

function  TCCRInterfaceElement.getColor: TColor;
begin
  Result := fData.Color;
end;

function TCCRInterfaceElement.getDescription: String;
begin
  Result := fData.Descr;
end;

function  TCCRInterfaceElement.getFontColor: TColor;
begin
  Result := fData.FontColor;
end;

function  TCCRInterfaceElement.getFontSize: Integer;
begin
  Result := fData.FontSize;
end;

function TCCRInterfaceElement.getName: String;
begin
  Result := fData.Name;
end;

function TCCRInterfaceElement.getProps: TCCRElementProps;
begin
  Result := fData.Props;
end;

procedure TCCRInterfaceElement.Restore(aStore: TOvcAbstractStore; const aSection: String);
var
  ieTag: String;
begin
  ieTag := 'IE.' + Name + '.';
  if cepColor in Props then
    Color := aStore.ReadInteger(aSection, ieTag + 'Color', Color);
  if cepFontColor in Props then
    FontColor := aStore.ReadInteger(aSection, ieTag + 'FontColor', FontColor);
  if cepFontSize in Props then
    FontSize := aStore.ReadInteger(aSection, ieTag + 'FontSize', FontSize);
end;

procedure TCCRInterfaceElement.Save(aStore: TOvcAbstractStore; const aSection: String);
var
  ieTag: String;
begin
  ieTag := 'IE.' + Name + '.';
  if cepColor in Props then
    aStore.WriteInteger(aSection, ieTag + 'Color', Color);
  if cepFontColor in Props then
    aStore.WriteInteger(aSection, ieTag + 'FontColor', FontColor);
  if cepFontSize in Props then
    aStore.WriteInteger(aSection, ieTag + 'FontSize', FontSize);
end;

procedure TCCRInterfaceElement.setColor(const aColor: TColor);
begin
  fData.Color := aColor;
end;

procedure TCCRInterfaceElement.setFontColor(const aColor: TColor);
begin
  fData.FontColor := aColor;
end;

procedure TCCRInterfaceElement.setFontSize(const aSize: Integer);
begin
  fData.FontSize := aSize;
end;

////////////////////////////// TCCRInterfaceElements \\\\\\\\\\\\\\\\\\\\\\\\\\\

function TCCRInterfaceElements.Add(const aData: TCCRInterfaceElementDef): TCCRInterfaceElement;
begin
  Result := TCCRInterfaceElement(inherited Add);
  if Assigned(Result) then
    Result.AssignDef(aData);
end;

function TCCRInterfaceElements.getItem(anIndex: Integer): TCCRInterfaceElement;
begin
  Result := TCCRInterfaceElement(inherited GetItem(anIndex));
end;

procedure TCCRInterfaceElements.RestoreDefaults;
var
  i, n: Integer;
begin
  BeginUpdate;
  try
    Clear;
    n := Length(InterfaceElementDefs) - 1;
    for i:=0 to n do
      Add(InterfaceElementDefs[i]);
  finally
    EndUpdate;
  end;
end;

/////////////////////////////////// TCCROptions \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCROptions.Create;
begin
  inherited;
  fCDC1Bmp := 'CDC1.bmp';
  fCDC2Bmp := 'CDC2.bmp';
  fInterfaceElements := TCCRInterfaceElements.Create(TCCRInterfaceElement);
  RestoreDefaults;
end;

destructor TCCROptions.Destroy;
begin
  FreeAndNil(fInterfaceElements);
  inherited;
end;

procedure TCCROptions.Apply;
var
  i: Integer;
  Method: TMethod;
begin
  Application.ShowHint   := ShowHints;
  Broker.RPCBroker.RPCTimeLimit := RPCTimeout;

  for i:=Screen.CustomFormCount-1 downto 0 do
    begin
      Method.Code := Screen.CustomForms[i].MethodAddress('ApplyOptions');
      if Method.Code <> nil then
        begin
          Method.Data := Screen.CustomForms[i];
          TCCRApplyOptionsEvent(Method);
        end;
    end;
end;

function TCCROptions.Edit: Boolean;
begin
  with TFormOptions.Create(nil) do
    try
      Result := (ShowModal = mrOK);
    finally
      Free;
    end;
end;

procedure TCCROptions.Restore(aState: TOvcPersistentState);
var
  tmpForm: TFormOptions;
  i, n: Integer;
begin
  tmpForm := nil;
  if not Assigned(aState) then
    begin
      tmpForm := TFormOptions.Create(nil);
      aState := tmpForm.vxsOptions;
    end;
  try
    aState.Storage.Open;
    try
      aState.RestoreState(Self, SECTION);
      n := IE.Count - 1;
      for i:=0 to n do
        IE[i].Restore(aState.Storage, SECTION);
    finally
      aState.Storage.Close;
    end;
  finally
    tmpForm.Free;
  end;
end;

procedure TCCROptions.RestoreDefaults;
begin
  MaxPatientCount := 300;
  RPCTimeout      := 60;
  ShowHints       := True;

  IE.RestoreDefaults;
end;

procedure TCCROptions.Save(aState: TOvcPersistentState);
var
  tmpForm: TFormOptions;
  i, n: Integer;
begin
  tmpForm := nil;
  if not Assigned(aState) then
    begin
      tmpForm := TFormOptions.Create(nil);
      aState := tmpForm.vxsOptions;
    end;
  try
    aState.Storage.Open;
    try
      aState.SaveState(Self, SECTION);
      n := IE.Count - 1;
      for i:=0 to n do
        IE[i].Save(aState.Storage, SECTION);
    finally
      aState.Storage.Close;
    end;
  finally
    tmpForm.Free;
  end;
end;

//////////////////////////////////// TFormOptions \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TFormOptions.Create(AOwner: TComponent);
begin
  inherited;
  pcMain.ActivePageIndex := 0;
  UpdateAllControls;
end;

procedure TFormOptions.AfterConstruction;
begin
  inherited;
  if CCRScreenReaderActive then
    stxtMaxPatientCount.TabStop := true;
end;

procedure TFormOptions.FormShow(Sender: TObject);
begin
  if stxtMaxPatientCount.TabStop then
    stxtMaxPatientCount.SetFocus;
end;

procedure TFormOptions.acDefaultExecute(Sender: TObject);
begin
  CCROptions.RestoreDefaults;
  UpdateAllControls;
end;

procedure TFormOptions.cmbxColorSelect(Sender: TObject);
begin
  if lbIE.ItemIndex >= 0 then
    with TCCRInterfaceElement(lbIE.Items.Objects[lbIE.ItemIndex]) do
      begin
        Color := cmbxColor.Selected;
        memoSample.Color := Color;
      end;
end;

procedure TFormOptions.cmbxFontColorSelect(Sender: TObject);
begin
  if lbIE.ItemIndex >= 0 then
    with TCCRInterfaceElement(lbIE.Items.Objects[lbIE.ItemIndex]) do
      begin
        FontColor := cmbxFontColor.Selected;
        memoSample.Font.Color := FontColor;
      end;
end;

procedure TFormOptions.cmbxFontSizeChange(Sender: TObject);
begin
  if lbIE.ItemIndex >= 0 then
    with TCCRInterfaceElement(lbIE.Items.Objects[lbIE.ItemIndex]) do
      begin
        FontSize := StrToIntDef(cmbxFontSize.Text, FontSize);
        cmbxFontSize.Text := IntToStr(FontSize);
        memoSample.Font.Size := FontSize;
      end;
end;

function TFormOptions.FetchControlValues(const Silent: Boolean): Boolean;
var
  tmpInt: Integer;
  msg: String;
  errCtrl: TWinControl;
begin
  msg := '';
  errCtrl := nil;

  with CCROptions do
    begin
      tmpInt := StrToIntDef(edMaxPatientCount.Text, -1);
      if (tmpInt < StrToInt(edMaxPatientCount.RangeLo)) or
         (tmpInt > StrToInt(edMaxPatientCount.RangeHi))then
        begin
          msg := msg + #13 + stxtMaxPatientCount.Caption;
          if not Assigned(errCtrl) then
            errCtrl := edMaxPatientCount;
        end
      else
        MaxPatientCount := tmpInt;

      tmpInt := StrToIntDef(cmbxTimeout.Text, -1);
      if tmpInt < MINRPCTIMEOUT then
        begin
          msg := msg + #13 + lblTimeout.Caption;
          if not Assigned(errCtrl) then
            errCtrl := cmbxTimeout;
        end
      else
        RPCTimeout := tmpInt;

      ShowHints := cbShowHints.Checked;
    end;

  if Assigned(errCtrl) then
    begin
      MessageDlg508('', RSC00230 + #13 + msg, mtError, [mbOK], 0);
      ActiveControl := errCtrl;
      Result := False;
    end
  else
    Result := True;
end;

procedure TFormOptions.lbIEClick(Sender: TObject);
begin
  UpdateIEControls(lbIE.ItemIndex);
end;

procedure TFormOptions.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  if ModalResult = mrOK then
    begin
      CanClose := FetchControlValues;
      if CanClose then
        begin
          CCROptions.Save(vxsOptions);
          CCROptions.Apply;
        end;
    end
  else
    begin
      CCROptions.Restore(vxsOptions);
      UpdateAllControls;
    end;
end;

procedure TFormOptions.UpdateAllControls;
var
  i, n: Integer;
begin

  with CCROptions do
    begin
      edMaxPatientCount.Text := IntToStr(MaxPatientCount);
      cmbxTimeout.Text       := IntToStr(RPCTimeout);
      cbShowHints.Checked    := ShowHints;
    end;

  //--- Repopulate the list-box of interface elements
  lbIE.Items.BeginUpdate;
  try
    lbIE.Clear;
    n := CCROptions.IE.Count - 1;
    for i:=0 to n do
      lbIE.AddItem(CCROptions.IE[i].Description, CCROptions.IE[i]);
    if n > 0 then
      lbIE.ItemIndex := 0;
  finally
    lbIE.Items.EndUpdate;
  end;

  //--- Update parameters of the current interface element
  UpdateIEControls(lbIE.ItemIndex);
end;


procedure TFormOptions.UpdateIEControls(const anIndex: Integer);
begin
  if anIndex >= 0 then
    with TCCRInterfaceElement(lbIE.Items.Objects[anIndex]) do
      begin
        cmbxColor.Selected := Color;
        cmbxFontColor.Selected := FontColor;
        cmbxFontSize.Text := IntToStr(FontSize);
        memoSample.Color := Color;
        memoSample.Font.Color := FontColor;
        memoSample.Font.Size := FontSize;
        //--- Background color
        cmbxColor.Enabled      := (cepColor in Props);
        cmbxColor.Color        := CtrlColor(cmbxColor.Enabled);
        lblColor.Enabled       := cmbxColor.Enabled;
        //--- Foreground color
        cmbxFontColor.Enabled  := (cepFontColor in Props);
        cmbxFontColor.Color    := CtrlColor(cmbxFontColor.Enabled);
        lblFontColor.Enabled   := cmbxFontColor.Enabled;
        //--- Font size
        cmbxFontSize.Enabled   := (cepFontSize in Props);
        lblFontSize.Enabled    := cmbxFontSize.Enabled;
        cmbxFontSize.Color     := CtrlColor(cmbxFontSize.Enabled);
      end
  else
    begin
      memoSample.Color      := CtrlColor(False);;

      cmbxColor.Color       := CtrlColor(False);
      cmbxColor.Enabled     := False;
      lblColor.Enabled      := False;

      cmbxFontColor.Color   := CtrlColor(False);
      cmbxFontColor.Enabled := False;
      lblFontColor.Enabled  := False;

      cmbxFontSize.Color    := CtrlColor(False);
      cmbxFontSize.Enabled  := False;
      lblFontSize.Enabled   := False;
    end;
end;

procedure TFormOptions.vxsStorageBeforeOpen(Sender: TObject);
begin
  vxsStorage.Broker := Broker;
end;

end.
